﻿''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
'Main.c
'
'Version 1.0   Mar 2010
'
'Go to http://geoffg.net/fancontroller.html for updates, errata and helpful notes
'
'    Copyright 2010 Geoff Graham - http://geoffg.net
'    This program is free software: you can redistribute it and/or modify it under the terms of the GNU General
'    Public License as published by the Free Software Foundation, either version 2 of the License, or (at your
'    option) any later version.
'
'    This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
'    implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
'    for more details.  You should have received a copy of the GNU General Public License along with this program.
'    If not, see <http://www.gnu.org/licenses/>.
'
'
'This is the main source file for the desktop application of the Fan Controller project.
'
'Development Environment
'    To compile this you need:
'     - Microsoft Visual Basic 2008 Express Version with Service Pack 1 (it is free) (www.microsoft.com)
'
'Explanation:
'    This code is not as neat as it could be.  It was ported from VB5 and the process turned out to be more complicated than I
'    anticipated.  This is in part due to the fact that VB 2008 does not have arrays of controls (as does VB6).  Another issue
'    was with the serial interface (see below).  This code should really be re written from the ground up to suit VB 2008.
'
'  #BeginRant
'     When I started this project I chose VB 2008 because there was a free version (good for readers) and I thought that development 
'     would be quick and easy because VB5 (which I was familiar with) was quick and easy.  Silly me...
'
'     To access a serial port in VB 2008 (aka VB.NET) you invoke the class IO.Ports.SerialPort
'     But the developer of this class chose to implement it in a separate thread.  This would be good if the serial port handled
'     lots of high speed data, but it doesn't - serial data is slow and does not tax the computer at all. So threads are not needed.
'
'     Anyway, because you read the data in a separate thread you cannot simply update controls (text boxes, etc) with the data received, 
'     you must use a delegate to invoke the control's update event.  None of this is explained in Microsoft's documentation so you are 
'     forced to trawl the Internet for some help.
'
'     Having figured this out I then discovered that the IO.Ports.SerialPort class has a bug which causes the serial port close function 
'     to intermittently hang when you use this technique (note that "intermittent" means many hours of debugging).  Microsoft recognise 
'     this issue but their database lists the issue as closed because it is "by design".  Amazing, they say that a bug is designed in!
'
'     A workaround exists involving the .BeginInvoke function but the serial port thread will then hold the focus preventing the main
'     form from interacting with the user.  So, that did not work either.  By now I had spent a good 40 hours on the problem digging 
'     through obscure Internet groups, deciphering Microsoft's poor documentation and trying endless code variations.
'
'     The real problem is the incredible complexity that Microsoft has created with VB 2008 (and the .NET approach generally).  It is
'     so complex that Microsoft cannot properly document it and a casual programmer cannot use it.  The whole thing is based on the 
'     assumption that you work for Microsoft, have had a huge amount of training and can make a direct phone call to the developers.
'
'     And this is not just an issue with the free version of VB 2008, it is in all aspects of .NET including the professional versions 
'     and C# and C++.  These are Microsoft's flagship products for software development and the basis of their future direction.
'  #EndRant
'
' In the end I used a very crude approach.  When data comes in from the serial port it is written to a variable called SerialData
' and another variable SerialDataReady is set to true.  tmrCheckForData is a timer which triggers once a second and in the timer event
' we look at SerialDataReady to see if there is something to process.  Because tmrCheckForData triggers only once a second the load on  
' the computer is very light, but it is a crude way to implement something so simple.
'
''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''


Imports System.Drawing
Imports System.Drawing.Drawing2D
Imports System.Windows.Forms


Public Class MainForm

    ' constants defining the location of the data in the configuration string sent by the fan controller
    Const F_DATA_LEN = 7
    Const F_DATA_START = 4
    Const F_MINPWR = 0
    Const F_SENSOR = 1
    Const F_MINTEMP = 2
    Const F_MAXTEMP = 3
    Const F_CANSTOP = 4
    Const F_ATYPE = 5
    Const F_BTYPE = 6
    Const D_TYPE_FAN_1 = F_DATA_START + (0 * F_DATA_LEN)
    Const D_TYPE_FAN_2 = F_DATA_START + (1 * F_DATA_LEN)
    Const D_TYPE_FAN_3 = F_DATA_START + (2 * F_DATA_LEN)
    Const D_TYPE_FAN_4 = F_DATA_START + (3 * F_DATA_LEN)

    ' the current configuration data received from, and sent to, the fan controller
    Public ConfigData(0 To F_DATA_START + (F_DATA_LEN * 4)) As Integer
    Const FAULTY_SENSOR = 127

    ' the current operational data set received from the fan controller (temperatures, fan speeds, etc)
    Dim CurrentFanData(0 To 15) As Integer

    Public CommPortCurrent As String = ""                           ' the current comm port
    Dim CommSelectBoxReady As Boolean = False                       ' used to prevent the combo box processing its change event

    Dim GraphFullScaleValue As Integer                              ' the scale used for the bar gralp (in RPM)

    ' various states of the communications startup process
    Const COMM_CLOSED = 0
    Const COMM_CONNECTED = 1
    Const COMM_COMMUNICATING = 2
    Const COMM_REQUESTED_CONFIG_1 = 3
    Const COMM_REQUESTED_CONFIG_2 = 4
    Const COMM_GOT_CONFIG = 5
    Const COMM_REQ_CLOSE = 6
    Public CommPortConnected As Integer = COMM_CLOSED

    Dim SerialData As String = ""                                   ' the string of data as received via the serial port
    Dim SerialDataReady As Boolean = False                          ' flag indicating that data has been received

    ' various elements used in drawing the fan speed bar graphs
    Dim g As System.Drawing.Graphics
    Dim pen1 As New System.Drawing.Pen(Color.Blue, 8)
    Dim pen2 As New System.Drawing.Pen(Color.White, 8)
    Dim pen3 As New System.Drawing.Pen(Me.BackColor, 8)

    ' the infamous serial port
    Public WithEvents serialPort As New IO.Ports.SerialPort

    ' this triggers before the main form is loaded.  
    ' It is used to retrieve the last comm port used and tries to open it.  From the users perspective the application starts up
    ' connected to the fan controller
    Private Sub MainForm_Load(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles MyBase.Load
        Dim i As Integer
        SerialData = ""
        SerialDataReady = False

        GraphFullScale.Text = My.Settings.GraphFullScale
        GraphFullScaleValue = Val(My.Settings.GraphFullScale)

        For i = 0 To My.Computer.Ports.SerialPortNames.Count - 1
            If My.Computer.Ports.SerialPortNames(i) = My.Settings.LastComPort Then CommPortCurrent = My.Computer.Ports.SerialPortNames(i)
        Next

        OpenCom()
        EnableReadouts()

    End Sub


    ' Get data from the serial port
    Private Sub DataReceived(ByVal sender As Object, ByVal e As System.IO.Ports.SerialDataReceivedEventArgs) Handles serialPort.DataReceived
        Dim i As Integer
        Static Dim s As String = ""

        For i = 0 To serialPort.BytesToRead
            Try
                ' for some reason we cannot just get the data from the port.  We have to read it one character at a time and then go
                ' through this messy conversion process.  More unexplained Microsoft nonsense.
                s = s & Chr(Val(serialPort.ReadChar))
            Catch ex As Exception
                Exit Sub
            End Try
        Next

        ' accumulate data until we have a full line
        If s.Contains(vbLf) Then
            SerialData = s
            SerialDataReady = True
            s = ""
        End If

    End Sub


    ' check if data has arrived and process it if it has
    Private Sub tmrCheckForData_Tick(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles tmrCheckForData.Tick
        If SerialDataReady Then
            If CommPortConnected = COMM_GOT_CONFIG Then tmrCheckForData.Interval = 900
            ProcessSerialData()
            SerialDataReady = False
        Else
            tmrCheckForData.Interval = 100
        End If
    End Sub


    ' Process the data received from the serial port.  It is held in the global string SerialData
    Sub ProcessSerialData()
        Dim i As Integer

        ' process the standard data string from the fan controller
        If SerialData.StartsWith("FCD,") Then
            If CommPortConnected = COMM_CONNECTED Then
                CommStatus.Text = "  Found Fan Controller on " & CommPortCurrent & " - downloading setup"
                CommPortConnected = COMM_COMMUNICATING
            End If

            If CommPortConnected <> COMM_REQUESTED_CONFIG_1 And CommPortConnected <> COMM_REQUESTED_CONFIG_2 Then
                tmrTimeout.Enabled = False                              ' restart the timeout timer
                tmrTimeout.Enabled = True
            End If

            ' if we are in the startup phase we ask the fan controller for a dump of its configuration
            If CommPortConnected = COMM_COMMUNICATING Then
                My.Settings.LastComPort = CommPortCurrent
                CommPortConnected = COMM_REQUESTED_CONFIG_1
                SendToCOM("FCQ")
                Return
            End If

            ' we have everything we need, so display the data
            If CommPortConnected = COMM_GOT_CONFIG Then
                For i = 0 To 15
                    CurrentFanData(i) = GetDataItemFromString(SerialData, i)
                Next i

                ' test data
                'CurrentFanData(0) = 34
                'CurrentFanData(1) = 42
                'CurrentFanData(2) = 38
                'CurrentFanData(3) = 26
                'CurrentFanData(4) = 30
                'CurrentFanData(5) = 82
                'CurrentFanData(6) = 70
                'CurrentFanData(7) = 53
                'CurrentFanData(8) = 1045
                'CurrentFanData(9) = 1408
                'CurrentFanData(10) = 2558
                'CurrentFanData(11) = 2577
                'CurrentFanData(12) = 1800
                'CurrentFanData(13) = 1060
                'CurrentFanData(14) = 1440
                'CurrentFanData(15) = 1467

                ' Get the right string into the text box.  Either °C, °F or blank (if invalid or not setup).
                SensorVal_A.Text = If(ConfigData(0) = 2, Int((CurrentFanData(0) * 9) / 5 + 32) & " °F", CurrentFanData(0) & " °C")
                If ConfigData(0) = 0 Or CurrentFanData(0) = FAULTY_SENSOR Then SensorVal_A.Text = " "
                SensorVal_B.Text = If(ConfigData(1) = 2, Int((CurrentFanData(1) * 9) / 5 + 32) & " °F", CurrentFanData(1) & " °C")
                If ConfigData(1) = 0 Or CurrentFanData(1) = FAULTY_SENSOR Then SensorVal_B.Text = " "
                SensorVal_C.Text = If(ConfigData(2) = 2, Int((CurrentFanData(2) * 9) / 5 + 32) & " °F", CurrentFanData(2) & " °C")
                If ConfigData(2) = 0 Or CurrentFanData(2) = FAULTY_SENSOR Then SensorVal_C.Text = " "
                SensorVal_D.Text = If(ConfigData(3) = 2, Int((CurrentFanData(3) * 9) / 5 + 32) & " °F", CurrentFanData(3) & " °C")
                If ConfigData(3) = 0 Or CurrentFanData(3) = FAULTY_SENSOR Then SensorVal_D.Text = " "

                If ConfigData(0) <> 0 Then
                    If CurrentFanData(0) = FAULTY_SENSOR Then
                        SensorVal_A.BackColor = Color.Red
                    Else
                        SensorVal_A.BackColor = Color.White
                    End If
                Else
                    SensorVal_A.BackColor = Me.BackColor
                End If

                If ConfigData(1) <> 0 Then
                    If CurrentFanData(1) = FAULTY_SENSOR Then
                        SensorVal_B.BackColor = Color.Red
                    Else
                        SensorVal_B.BackColor = Color.White
                    End If
                Else
                    SensorVal_B.BackColor = Me.BackColor
                End If

                If ConfigData(2) <> 0 Then
                    If CurrentFanData(2) = FAULTY_SENSOR Then
                        SensorVal_C.BackColor = Color.Red
                    Else
                        SensorVal_C.BackColor = Color.White
                    End If
                Else
                    SensorVal_C.BackColor = Me.BackColor
                End If

                If ConfigData(3) <> 0 Then
                    If CurrentFanData(3) = FAULTY_SENSOR Then
                        SensorVal_D.BackColor = Color.Red
                    Else
                        SensorVal_D.BackColor = Color.White
                    End If
                Else
                    SensorVal_D.BackColor = Me.BackColor
                End If


                DrawBar(0, CurrentFanData(8), True, CurrentFanData(4))
                If ConfigData(D_TYPE_FAN_1 + F_ATYPE) = 1 Then DrawBar(0, CurrentFanData(4), False, 0)
                DrawBar(1, CurrentFanData(9), True, CurrentFanData(4))
                If ConfigData(D_TYPE_FAN_1 + F_BTYPE) = 1 Then DrawBar(1, CurrentFanData(4), False, 0)
                DrawBar(2, CurrentFanData(10), True, CurrentFanData(5))
                If ConfigData(D_TYPE_FAN_2 + F_ATYPE) = 1 Then DrawBar(2, CurrentFanData(5), False, 0)
                DrawBar(3, CurrentFanData(11), True, CurrentFanData(5))
                If ConfigData(D_TYPE_FAN_2 + F_BTYPE) = 1 Then DrawBar(3, CurrentFanData(5), False, 0)
                DrawBar(4, CurrentFanData(12), True, CurrentFanData(6))
                If ConfigData(D_TYPE_FAN_3 + F_ATYPE) = 1 Then DrawBar(4, CurrentFanData(6), False, 0)
                DrawBar(5, CurrentFanData(13), True, CurrentFanData(6))
                If ConfigData(D_TYPE_FAN_3 + F_BTYPE) = 1 Then DrawBar(5, CurrentFanData(6), False, 0)
                DrawBar(6, CurrentFanData(14), True, CurrentFanData(7))
                If ConfigData(D_TYPE_FAN_4 + F_ATYPE) = 1 Then DrawBar(6, CurrentFanData(7), False, 0)
                DrawBar(7, CurrentFanData(15), True, CurrentFanData(7))
                If ConfigData(D_TYPE_FAN_4 + F_BTYPE) = 1 Then DrawBar(7, CurrentFanData(7), False, 0)

                My.Application.DoEvents()

            End If  ' CommPortConnected = COMM_GOT_CONFIG

        End If  ' Received "FCD"

        ' process the configuration data from the fan controller.  This is sent in response to a "FCQ" command
        ' from this software
        If SerialData.StartsWith("FCR,") And (CommPortConnected = COMM_REQUESTED_CONFIG_1 Or CommPortConnected = COMM_REQUESTED_CONFIG_2) Then
            tmrTimeout.Enabled = False                              ' restart the timeout timer
            tmrTimeout.Enabled = True

            On Error Resume Next
            ConfigForm.Close()
            On Error GoTo 0

            For i = 0 To F_DATA_START + (F_DATA_LEN * 4) - 1
                ConfigData(i) = GetDataItemFromString(SerialData, i)
            Next i

            EnableReadouts()
            FansAndSensorsToolStripMenuItem.Enabled = True
            CommStatus.Text = "  Connected to the Fan Controller on " & CommPortCurrent
            CommPortConnected = COMM_GOT_CONFIG
        End If  ' Received "FCR"


    End Sub


    ' open the serial port
    Public Function OpenCom() As Boolean

        If CommPortCurrent = "" Then Return False

        If serialPort.IsOpen Then
            Try
                serialPort.Close()
            Catch ex As Exception
                Return False
            End Try
        End If
        CommPortConnected = COMM_CLOSED
        CommStatus.Text = "  Opening " & CommPortCurrent

        Try
            With serialPort
                .PortName = CommPortCurrent
                .BaudRate = 9600
                .Parity = IO.Ports.Parity.None
                .DataBits = 8
                .StopBits = IO.Ports.StopBits.One
                .ReadTimeout = 10000
                .WriteTimeout = 10000
            End With
            serialPort.Open()
        Catch ex As Exception
            CommStatus.Text = "  Not connected"
            CommPortConnected = COMM_CLOSED
            MsgBox("Error opening  " & CommPortCurrent, 16, "Fan Controller")
            Return False
        End Try
        serialPort.DiscardInBuffer()
        serialPort.DiscardOutBuffer()

        CommPortConnected = COMM_CONNECTED
        CommStatus.Text = "  Opened " & CommPortCurrent & " -  Waiting for data"
        tmrTimeout.Enabled = True
        tmrCheckForData.Interval = 100
        tmrCheckForData.Enabled = True

        Return True

    End Function


    ' close the serial port
    Sub CloseComm()
        CommPortConnected = COMM_CLOSED
        tmrCheckForData.Enabled = False
        tmrTimeout.Enabled = False
        CommPortCurrent = ""
        CommStatus.Text = "  Disconnected"
        CommunicationsToolStripMenuItem.HideDropDown()
        ToolStripMenuItem1.HideDropDown()
        FansAndSensorsToolStripMenuItem.Enabled = False
        ConfigForm.Close()

        For i = 0 To F_DATA_START + (F_DATA_LEN * 4) - 1
            ConfigData(i) = 0
        Next

        SensorVal_A.Text = ""
        SensorVal_B.Text = ""
        SensorVal_C.Text = ""
        SensorVal_D.Text = ""

        EnableReadouts()

        For i = 0 To 7
            DrawBar(i, 0, False, 0)
        Next

        On Error Resume Next
        If serialPort.IsOpen Then
            serialPort.DiscardInBuffer()
            serialPort.DiscardOutBuffer()
            serialPort.Close()
        End If

    End Sub


    ' send something to the serial port
    Public Sub SendToCOM(ByVal s As String)
        Try
            serialPort.Write(s & vbCrLf)
        Catch ex As Exception
            MsgBox("Error sending data to the Fan Controller " & CommPortCurrent, 16, "Fan Controller")
            CloseComm()
        End Try
    End Sub


    ' the user has requested that the form be closed.  Close the serial port and dispose of everything
    Private Sub MainForm_FormClosing(ByVal sender As System.Object, ByVal e As System.Windows.Forms.FormClosingEventArgs) Handles MyBase.FormClosing
        On Error Resume Next
        CloseComm()
        pen1.Dispose()
        pen2.Dispose()
        pen3.Dispose()
        serialPort.Dispose()
    End Sub


    ' If no data has been received for 5 seconds this will trigger and display an error
    Private Sub tmrTimeout_Tick(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles tmrTimeout.Tick
        Dim s As String = CommPortCurrent

        If CommPortConnected = COMM_REQUESTED_CONFIG_1 Then
            SendToCOM("FCQ")
            CommPortConnected = COMM_REQUESTED_CONFIG_2
            CommStatus.Text = "Resending request for setup data"
        Else
            CloseComm()
            MsgBox("No data from the fan controller on port " & s, 16, "Fan Controller")
        End If
    End Sub


    ' draw the vertical bar for fan speed.  Includes displaying the speed as a number
    Private Sub DrawBar(ByVal Bar As Integer, ByVal Speed As Integer, ByVal RMPSensor As Boolean, ByVal Power As Integer)
        Dim h As Integer
        Dim s As String
        Dim c As Color
        Dim e As Boolean

        If RMPSensor Then
            s = vbCrLf & "RPM"
        Else
            s = "%" & vbCrLf & "power"
        End If

        If RMPSensor And Speed < 60 And Power <> 0 Then
            c = Color.Red
        Else
            c = Me.BackColor
        End If

        Select Case Bar
            Case 0
                e = PictureBox1.Visible
                g = PictureBox1.CreateGraphics
                If Not PictureBox1.Enabled Then Speed = 0
                lblSpeed1.Text = Speed & s
                lblSpeed1.BackColor = c
            Case 1
                e = PictureBox2.Visible
                g = PictureBox2.CreateGraphics
                If Not PictureBox2.Enabled Then Speed = 0
                lblSpeed2.Text = Speed & s
                lblSpeed2.BackColor = c
            Case 2
                e = PictureBox3.Visible
                g = PictureBox3.CreateGraphics
                If Not PictureBox3.Enabled Then Speed = 0
                lblSpeed3.Text = Speed & s
                lblSpeed3.BackColor = c
            Case 3
                e = PictureBox4.Visible
                g = PictureBox4.CreateGraphics
                If Not PictureBox4.Enabled Then Speed = 0
                lblSpeed4.Text = Speed & s
                lblSpeed4.BackColor = c
            Case 4
                e = PictureBox5.Visible
                g = PictureBox5.CreateGraphics
                If Not PictureBox5.Enabled Then Speed = 0
                lblSpeed5.Text = Speed & s
                lblSpeed5.BackColor = c
            Case 5
                e = PictureBox6.Visible
                g = PictureBox6.CreateGraphics
                If Not PictureBox6.Enabled Then Speed = 0
                lblSpeed6.Text = Speed & s
                lblSpeed6.BackColor = c
            Case 6
                e = PictureBox7.Visible
                g = PictureBox7.CreateGraphics
                If Not PictureBox7.Enabled Then Speed = 0
                lblSpeed7.Text = Speed & s
                lblSpeed7.BackColor = c
            Case 7
                e = PictureBox8.Visible
                g = PictureBox8.CreateGraphics
                If Not PictureBox8.Enabled Then Speed = 0
                lblSpeed8.Text = Speed & s
                lblSpeed8.BackColor = c
        End Select

        If RMPSensor Then
            h = (Speed * 86) / GraphFullScaleValue
        Else
            h = (Speed * 86) / 100
        End If
        If (h > 86) Then h = 86
        If e Then
            g.DrawRectangle(pen2, 2, 0, 8, 90)
            If Speed <> 0 Then g.DrawRectangle(pen1, 2, 90 - h, 8, 90)
        Else
            g.DrawRectangle(pen3, 2, 0, 8, 90)
        End If
    End Sub


    ' load the configuration form
    ' this involves writing the current configuration data into the various controls (or, why did Microsoft not implement 
    ' arrays of controls as in VB6).
    Sub LoadConfigForm()

        ConfigForm.SensorType1.SelectedIndex = ConfigData(0)
        ConfigForm.SensorType2.SelectedIndex = ConfigData(1)
        ConfigForm.SensorType3.SelectedIndex = ConfigData(2)
        ConfigForm.SensorType4.SelectedIndex = ConfigData(3)

        ConfigForm.FanType1.SelectedIndex = ConfigData(D_TYPE_FAN_1 + F_ATYPE)
        ConfigForm.FanType2.SelectedIndex = ConfigData(D_TYPE_FAN_1 + F_BTYPE)
        ConfigForm.FanType3.SelectedIndex = ConfigData(D_TYPE_FAN_2 + F_ATYPE)
        ConfigForm.FanType4.SelectedIndex = ConfigData(D_TYPE_FAN_2 + F_BTYPE)
        ConfigForm.FanType5.SelectedIndex = ConfigData(D_TYPE_FAN_3 + F_ATYPE)
        ConfigForm.FanType6.SelectedIndex = ConfigData(D_TYPE_FAN_3 + F_BTYPE)
        ConfigForm.FanType7.SelectedIndex = ConfigData(D_TYPE_FAN_4 + F_ATYPE)
        ConfigForm.FanType8.SelectedIndex = ConfigData(D_TYPE_FAN_4 + F_BTYPE)

        ConfigForm.FanControl1.SelectedIndex = ConfigData(D_TYPE_FAN_1 + F_SENSOR)
        ConfigForm.FanControl2.SelectedIndex = ConfigData(D_TYPE_FAN_2 + F_SENSOR)
        ConfigForm.FanControl3.SelectedIndex = ConfigData(D_TYPE_FAN_3 + F_SENSOR)
        ConfigForm.FanControl4.SelectedIndex = ConfigData(D_TYPE_FAN_4 + F_SENSOR)

        ConfigForm.FanMinSpeed1.SelectedIndex = ConfigData(D_TYPE_FAN_1 + F_MINPWR) / 5
        ConfigForm.FanMinSpeed2.SelectedIndex = ConfigData(D_TYPE_FAN_2 + F_MINPWR) / 5
        ConfigForm.FanMinSpeed3.SelectedIndex = ConfigData(D_TYPE_FAN_3 + F_MINPWR) / 5
        ConfigForm.FanMinSpeed4.SelectedIndex = ConfigData(D_TYPE_FAN_4 + F_MINPWR) / 5

        ConfigForm.FanMinTemp1.SelectedIndex = ConfigData(D_TYPE_FAN_1 + F_MINTEMP) / 2
        ConfigForm.FanMinTemp2.SelectedIndex = ConfigData(D_TYPE_FAN_2 + F_MINTEMP) / 2
        ConfigForm.FanMinTemp3.SelectedIndex = ConfigData(D_TYPE_FAN_3 + F_MINTEMP) / 2
        ConfigForm.FanMinTemp4.SelectedIndex = ConfigData(D_TYPE_FAN_4 + F_MINTEMP) / 2

        ConfigForm.FanMaxTemp1.SelectedIndex = ConfigData(D_TYPE_FAN_1 + F_MAXTEMP) / 2
        ConfigForm.FanMaxTemp2.SelectedIndex = ConfigData(D_TYPE_FAN_2 + F_MAXTEMP) / 2
        ConfigForm.FanMaxTemp3.SelectedIndex = ConfigData(D_TYPE_FAN_3 + F_MAXTEMP) / 2
        ConfigForm.FanMaxTemp4.SelectedIndex = ConfigData(D_TYPE_FAN_4 + F_MAXTEMP) / 2

        ConfigForm.FanAllowZero1.Checked = ConfigData(D_TYPE_FAN_1 + F_CANSTOP)
        ConfigForm.FanAllowZero2.Checked = ConfigData(D_TYPE_FAN_2 + F_CANSTOP)
        ConfigForm.FanAllowZero3.Checked = ConfigData(D_TYPE_FAN_3 + F_CANSTOP)
        ConfigForm.FanAllowZero4.Checked = ConfigData(D_TYPE_FAN_4 + F_CANSTOP)

    End Sub


    ' enable the various controls (temperature and fan speed readouts) on the main form
    ' again - why do we not have arrays of controls Mr Gates?
    Public Sub EnableReadouts()
        lblSpeed1.Visible = ConfigData(D_TYPE_FAN_1 + F_ATYPE)
        lblSpeed2.Visible = ConfigData(D_TYPE_FAN_1 + F_BTYPE)
        lblSpeed3.Visible = ConfigData(D_TYPE_FAN_2 + F_ATYPE)
        lblSpeed4.Visible = ConfigData(D_TYPE_FAN_2 + F_BTYPE)
        lblSpeed5.Visible = ConfigData(D_TYPE_FAN_3 + F_ATYPE)
        lblSpeed6.Visible = ConfigData(D_TYPE_FAN_3 + F_BTYPE)
        lblSpeed7.Visible = ConfigData(D_TYPE_FAN_4 + F_ATYPE)
        lblSpeed8.Visible = ConfigData(D_TYPE_FAN_4 + F_BTYPE)

        lblBar1.Enabled = ConfigData(D_TYPE_FAN_1 + F_ATYPE)
        lblBar2.Enabled = ConfigData(D_TYPE_FAN_1 + F_BTYPE)
        lblBar3.Enabled = ConfigData(D_TYPE_FAN_2 + F_ATYPE)
        lblBar4.Enabled = ConfigData(D_TYPE_FAN_2 + F_BTYPE)
        lblBar5.Enabled = ConfigData(D_TYPE_FAN_3 + F_ATYPE)
        lblBar6.Enabled = ConfigData(D_TYPE_FAN_3 + F_BTYPE)
        lblBar7.Enabled = ConfigData(D_TYPE_FAN_4 + F_ATYPE)
        lblBar8.Enabled = ConfigData(D_TYPE_FAN_4 + F_BTYPE)

        PictureBox1.Visible = ConfigData(D_TYPE_FAN_1 + F_ATYPE)
        PictureBox2.Visible = ConfigData(D_TYPE_FAN_1 + F_BTYPE)
        PictureBox3.Visible = ConfigData(D_TYPE_FAN_2 + F_ATYPE)
        PictureBox4.Visible = ConfigData(D_TYPE_FAN_2 + F_BTYPE)
        PictureBox5.Visible = ConfigData(D_TYPE_FAN_3 + F_ATYPE)
        PictureBox6.Visible = ConfigData(D_TYPE_FAN_3 + F_BTYPE)
        PictureBox7.Visible = ConfigData(D_TYPE_FAN_4 + F_ATYPE)
        PictureBox8.Visible = ConfigData(D_TYPE_FAN_4 + F_BTYPE)

        SensorVal_A.BackColor = Color.White
        If ConfigData(0) = 0 Then SensorVal_A.BackColor = Me.BackColor
        SensorVal_A.Enabled = ConfigData(0)

        SensorVal_B.BackColor = Color.White
        If ConfigData(1) = 0 Then SensorVal_B.BackColor = Me.BackColor
        SensorVal_B.Enabled = ConfigData(1)

        SensorVal_C.BackColor = Color.White
        If ConfigData(2) = 0 Then SensorVal_C.BackColor = Me.BackColor
        SensorVal_C.Enabled = ConfigData(2)

        SensorVal_D.BackColor = Color.White
        If ConfigData(3) = 0 Then SensorVal_D.BackColor = Me.BackColor
        SensorVal_D.Enabled = ConfigData(3)

        lblSensorA.Enabled = ConfigData(0)
        lblSensorB.Enabled = ConfigData(1)
        lblSensorC.Enabled = ConfigData(2)
        lblSensorD.Enabled = ConfigData(3)

    End Sub


    ' Handler for when the user clicks on the Fan Configuration menu item
    Private Sub FansAndSensorsToolStripMenuItem_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles FansAndSensorsToolStripMenuItem.Click
        LoadConfigForm()
        ConfigForm.OK.Enabled = (CommPortConnected = COMM_GOT_CONFIG)
        ConfigForm.Apply.Enabled = (CommPortConnected = COMM_GOT_CONFIG)
        ConfigForm.ShowDialog(Me)
    End Sub

    ' Handler for when the user clicks on the About menu item
    Private Sub AboutToolStripMenuItem_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles AboutToolStripMenuItem.Click
        About.ShowDialog(Me)
    End Sub

    ' Handler for when the user opens the communications ports combo box
    ' the main job is to load the combo box with a list of the currently available serial ports
    Private Sub CommBoxSetup(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles CommunicationsToolStripMenuItem.Click, ToolStripMenuItem1.DropDownOpened
        Dim i As Integer
        Dim j As Integer

        CommSelectBoxReady = False
        j = 0
        CommSelectBox.Items.Clear()
        If CommPortConnected = COMM_CLOSED Then
            CommSelectBox.Items.Add("SELECT")
        Else
            CommSelectBox.Items.Add("CLOSE")

        End If

        ' step through the available serial ports and add them to the list
        For i = 0 To My.Computer.Ports.SerialPortNames.Count - 1
            CommSelectBox.Items.Add(My.Computer.Ports.SerialPortNames(i))
            If CommPortConnected <> COMM_CLOSED And My.Computer.Ports.SerialPortNames(i) = CommPortCurrent Then j = i + 1
        Next
        CommSelectBox.SelectedIndex = j
        CommSelectBoxReady = True
    End Sub


    ' Handler for when the user selects an item from the combo box
    ' it closes whatever port is currently open and opens the new selection
    Private Sub CommSelectBox_SelectedIndexChanged(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles CommSelectBox.SelectedIndexChanged
        If Not CommSelectBoxReady Then Exit Sub
        If CommSelectBox.Text = "CLOSE" Then
            CloseComm()
        End If

        If CommSelectBox.SelectedIndex < 1 Or CommPortCurrent = CommSelectBox.Text Then Exit Sub
        CloseComm()
        CommPortCurrent = CommSelectBox.Text
        CommPortCurrent = CommSelectBox.Text
        CommunicationsToolStripMenuItem.HideDropDown()
        ToolStripMenuItem1.HideDropDown()
        OpenCom()
    End Sub


    ' retrieve a specific field of the data from the fan controller 
    ' s = string of characters from the fan controller
    ' p = the field number we want (starting with zero as the first numeric item)
    Private Function GetDataItemFromString(ByVal s As String, ByVal p As Integer) As Integer
        Dim i As Integer
        Dim j As Integer

        On Error GoTo Abort
        j = 0
        For i = 0 To p
            j = s.IndexOf(",", j + 1)
            If j = -1 Then Return 0
        Next i
        i = s.IndexOf(",", j + 1)

        If i = -1 Then
            Return CInt(s.Substring(j + 1))
        Else
            Return CInt(s.Substring(j + 1, i - j))
        End If

Abort:
        Debug.WriteLine("GetDataItemFromString error: p = " & p & "  s = " & s)
        Return 0
    End Function

    '    Private Sub FanScaleCombo_SelectedIndexChanged(ByVal sender As Object, ByVal e As System.EventArgs)
    '       My.Settings.GraphFullScale = GraphFullScale.Text
    '       ToolStripMenuItem1.HideDropDown()
    '   End Sub


    Private Sub GraphFullScale_KeyPress(ByVal sender As Object, ByVal e As System.Windows.Forms.KeyPressEventArgs) Handles GraphFullScale.KeyPress
        If Not Char.IsDigit(e.KeyChar) And Not Char.IsControl(e.KeyChar) Then e.Handled = True
    End Sub

    Private Sub GraphFullScale_LostFocus(ByVal sender As Object, ByVal e As System.EventArgs) Handles GraphFullScale.LostFocus
        If Val(GraphFullScale.Text) < 100 Then GraphFullScale.Text = 100
        If Val(GraphFullScale.Text) > 10000 Then GraphFullScale.Text = 10000
        My.Settings.GraphFullScale = GraphFullScale.Text
        GraphFullScaleValue = Val(GraphFullScale.Text)
    End Sub
End Class
